#!/bin/bash

# Change to script directory
cd "$(dirname "$0")"

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Function to print colored output
print_header() {
    echo -e "${BLUE}============================================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}============================================================${NC}"
}

print_success() {
    echo -e "${GREEN}[OK]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_info() {
    echo -e "${CYAN}[INFO]${NC} $1"
}

clear
echo ""
print_header "         WhatsApp Server - ONE-CLICK INSTALLER"
echo ""
echo "This script will automatically:"
echo "  - Check if Node.js is installed"
echo "  - Install all required dependencies"
echo "  - Start the WhatsApp server"
echo ""
echo "Please wait..."
echo ""
print_header ""
echo ""

# Step 0: Auto-clean session for fresh start
echo "[Step 0/4] Checking session status..."
if [ -d ".wwebjs_auth" ]; then
    print_info "Found existing session folder."
    print_info "Deleting old session for fresh start..."
    rm -rf .wwebjs_auth 2>/dev/null
    print_success "Old session deleted! Will create new session."
else
    print_success "No existing session found. Will create fresh session."
fi
echo ""

# Step 1: Check Node.js
echo "[Step 1/4] Checking Node.js..."
if ! command -v node &> /dev/null; then
    print_error "Node.js is NOT installed!"
    echo ""
    echo "Please install Node.js first:"
    echo "  Ubuntu/Debian: sudo apt-get install nodejs npm"
    echo "  CentOS/RHEL:   sudo yum install nodejs npm"
    echo "  macOS:         brew install node"
    echo "  Or download from: https://nodejs.org/"
    echo ""
    echo "After installation, run this script again."
    echo ""
    read -p "Press Enter to exit..."
    exit 1
fi

NODE_VERSION=$(node --version)
print_success "Node.js $NODE_VERSION found!"
echo ""

# Step 2: Check package.json
echo "[Step 2/5] Checking project files..."
if [ ! -f "package.json" ]; then
    print_error "package.json not found!"
    echo "        Make sure you're running this from the correct folder."
    echo "        Current folder: $(pwd)"
    echo ""
    read -p "Press Enter to exit..."
    exit 1
fi
print_success "Project files found!"
echo ""

# Step 3: Install dependencies (only if needed)
echo "[Step 3/5] Checking dependencies..."
if [ ! -d "node_modules" ]; then
    echo "Dependencies not installed. Installing now..."
    echo "This will take 2-5 minutes. Please wait..."
    echo ""

    # Clean install
    [ -f "package-lock.json" ] && rm -f package-lock.json 2>/dev/null

    # Install
    npm install --no-audit --no-fund

    if [ $? -eq 0 ]; then
        echo ""
        print_success "Dependencies installed successfully!"

        # Verify installation
        if [ -d "node_modules/whatsapp-web.js" ]; then
            print_success "whatsapp-web.js installed"
        fi
        if [ -d "node_modules/express" ]; then
            print_success "express installed"
        fi
        echo ""
    else
        echo ""
        print_warning "Standard installation failed. Trying alternative method..."
        npm install --legacy-peer-deps --no-audit --no-fund

        if [ $? -eq 0 ]; then
            print_success "Dependencies installed with legacy mode!"
            echo ""
        else
            print_error "Installation failed!"
            echo ""
            echo "Please try:"
            echo "1. Check internet connection"
            echo "2. Run with sudo if permission denied"
            echo "3. Manually run: npm install"
            echo ""
            read -p "Press Enter to exit..."
            exit 1
        fi
    fi
else
    print_success "Dependencies already installed!"
    echo ""
fi

# Step 4: Create directories
echo "[Step 4/5] Creating required directories..."
[ ! -d "uploads" ] && mkdir -p uploads 2>/dev/null
[ ! -d ".wwebjs_auth" ] && mkdir -p .wwebjs_auth 2>/dev/null
print_success "Directories ready!"
echo ""

# Step 5: Final preparation
echo "[Step 5/5] Preparing to start server..."
if [ -d ".wwebjs_auth/session" ]; then
    print_info "Will use existing WhatsApp session."
else
    print_info "Will create new WhatsApp session."
fi
print_success "Ready to start!"
echo ""

# Success message
print_header "                    SETUP COMPLETE!"
echo ""
echo "Everything is ready! Starting the server now..."
echo ""
echo "IMPORTANT INSTRUCTIONS:"
echo "  1. Wait for QR code to appear below"
echo "  2. Open WhatsApp on your phone"
echo "  3. Go to: Settings > Linked Devices"
echo "  4. Tap: Link a Device"
echo "  5. Scan the QR code that will appear"
echo ""
echo "After scanning:"
echo "  - Go to your web interface"
echo "  - Select \"Self-Hosted\""
echo "  - Click \"Connect\""
echo "  - Start sending messages!"
echo ""
echo "Server URL: http://localhost:3000"
echo ""
echo "Press Ctrl+C to stop the server anytime"
echo ""
print_header ""
echo ""

sleep 2

# Give option to diagnose first (auto-skip after 3 seconds)
echo ""
echo "Do you want to run diagnostics first? (Y/N)"
echo "Auto-starting server in 3 seconds..."
echo ""

read -t 3 -n 1 -r REPLY
if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo ""
    echo "Running diagnostics..."
    echo ""
    node diagnose.js
    echo ""
    read -p "Press Enter to continue starting the server..."
    echo ""
else
    echo ""
    echo "Skipping diagnostics..."
    echo ""
fi

# Start the server
echo "Starting server..."
echo ""
echo "IMPORTANT: When you close this terminal, the server will stop automatically."
echo ""
echo "If stuck at \"Initializing WhatsApp Web.js client...\" for more than 2 minutes:"
echo "  1. Open browser to: http://localhost:3000/qr"
echo "  2. You should see the QR code there!"
echo "  3. If blank, wait 30 seconds and refresh the page"
echo ""
echo "Alternative: If QR code never appears, try:"
echo "  - Close this terminal and run again (fresh session)"
echo "  - See FIX-QR-ISSUE.md for solutions"
echo ""
print_header ""
echo ""

# Function to handle cleanup on exit
cleanup() {
    echo ""
    print_header "Cleaning up..."
    echo ""

    print_info "Stopping Node.js processes..."
    pkill -f "node server.js" 2>/dev/null
    if [ $? -eq 0 ]; then
        print_success "Node.js stopped."
    else
        print_info "No Node.js processes to stop."
    fi

    print_success "Cleanup complete."
    echo ""
    print_header "Server has stopped."
    echo ""
    echo "To restart: Run this script again"
    echo "Session will be cleaned on next run for fresh start."
    echo ""
    read -p "Press Enter to exit..."
    exit 0
}

# Trap Ctrl+C and cleanup
trap cleanup SIGINT SIGTERM

# Start server
node server.js

# If server stops normally, cleanup
cleanup
