#!/bin/bash

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

print_header() {
    echo -e "${BLUE}============================================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}============================================================${NC}"
}

print_success() { echo -e "${GREEN}[OK]${NC} $1"; }
print_error() { echo -e "${RED}[ERROR]${NC} $1"; }
print_warning() { echo -e "${YELLOW}[WARNING]${NC} $1"; }
print_info() { echo -e "${CYAN}[INFO]${NC} $1"; }

# Change to script directory
cd "$(dirname "$0")"

clear
echo ""
print_header "  WhatsApp Server - FULLY AUTOMATIC SETUP"
echo ""
echo "This script will automatically:"
echo "  1. Check and install Node.js if needed"
echo "  2. Install all dependencies"
echo "  3. Create required folders"
echo "  4. Set up everything for you"
echo ""
echo "Please wait..."
echo ""

# Detect OS
OS="unknown"
if [[ "$OSTYPE" == "linux-gnu"* ]]; then
    OS="linux"
    if [ -f /etc/debian_version ]; then
        DISTRO="debian"
    elif [ -f /etc/redhat-release ]; then
        DISTRO="redhat"
    fi
elif [[ "$OSTYPE" == "darwin"* ]]; then
    OS="mac"
fi

# Step 1: Check/Install Node.js
echo "[1/6] Checking Node.js installation..."
if ! command -v node &> /dev/null; then
    print_warning "Node.js is NOT installed. Installing automatically..."
    echo ""

    if [ "$OS" == "linux" ]; then
        if [ "$DISTRO" == "debian" ]; then
            print_info "Detected Debian/Ubuntu system"
            echo "Installing Node.js..."

            # Try to install without sudo first
            if command -v apt-get &> /dev/null; then
                sudo apt-get update -qq
                sudo apt-get install -y nodejs npm

                if [ $? -eq 0 ]; then
                    print_success "Node.js installed successfully!"
                else
                    print_error "Installation failed. Please run with sudo."
                    exit 1
                fi
            fi

        elif [ "$DISTRO" == "redhat" ]; then
            print_info "Detected RedHat/CentOS system"
            sudo yum install -y nodejs npm

        else
            print_warning "Unknown Linux distribution"
            echo "Please install Node.js manually:"
            echo "  Visit: https://nodejs.org/"
            exit 1
        fi

    elif [ "$OS" == "mac" ]; then
        print_info "Detected macOS system"

        # Check if Homebrew is installed
        if command -v brew &> /dev/null; then
            print_info "Installing Node.js via Homebrew..."
            brew install node

            if [ $? -eq 0 ]; then
                print_success "Node.js installed successfully!"
            else
                print_error "Installation failed"
                exit 1
            fi
        else
            print_warning "Homebrew not found"
            echo "Please install Homebrew first:"
            echo "  /bin/bash -c \"\$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)\""
            echo ""
            echo "Or install Node.js manually from: https://nodejs.org/"
            exit 1
        fi
    else
        print_error "Unsupported operating system"
        echo "Please install Node.js manually from: https://nodejs.org/"
        exit 1
    fi

    # Verify installation
    sleep 2
    if ! command -v node &> /dev/null; then
        print_error "Node.js installation failed"
        echo "Please install manually from: https://nodejs.org/"
        exit 1
    fi
fi

NODE_VERSION=$(node --version)
print_success "Node.js found: $NODE_VERSION"
echo ""

# Step 2: Check npm
echo "[2/6] Checking npm..."
if ! command -v npm &> /dev/null; then
    print_error "npm not found!"
    echo "Please reinstall Node.js from: https://nodejs.org/"
    exit 1
fi

NPM_VERSION=$(npm --version)
print_success "npm found: v$NPM_VERSION"
echo ""

# Step 3: Verify package.json
echo "[3/6] Checking project files..."
if [ ! -f "package.json" ]; then
    print_error "package.json not found!"
    echo "Current directory: $(pwd)"
    exit 1
fi
print_success "package.json found"
echo ""

# Step 4: Clean previous installation
echo "[4/6] Cleaning previous installation..."
[ -d "node_modules" ] && rm -rf node_modules
[ -f "package-lock.json" ] && rm -f package-lock.json
print_success "Cleanup complete"
echo ""

# Step 5: Create directories
echo "[5/6] Creating required directories..."
mkdir -p uploads
mkdir -p .wwebjs_auth
chmod 755 uploads 2>/dev/null
chmod 755 .wwebjs_auth 2>/dev/null
print_success "Directories ready"
echo ""

# Step 6: Install dependencies
echo "[6/6] Installing dependencies automatically..."
echo "This will take 2-5 minutes. Please do NOT interrupt!"
echo ""

# Clear cache
npm cache clean --force > /dev/null 2>&1

# Install dependencies
npm install --yes --no-audit --no-fund --loglevel=error

if [ $? -eq 0 ]; then
    echo ""
    print_header "  SUCCESS! Everything is ready!"
    echo ""
    echo "Your WhatsApp server is now installed and ready to use."
    echo ""
    echo "Next step: Run './START.sh' to start the server"
    echo ""
    print_header ""
    echo ""

    # Create START.sh shortcut
    if [ ! -f "START.sh" ]; then
        cat > START.sh << 'EOF'
#!/bin/bash
cd "$(dirname "$0")"
./run.sh
EOF
        chmod +x START.sh
    fi

    # Make all scripts executable
    chmod +x *.sh 2>/dev/null
    print_success "All scripts are now executable"
    echo ""

    # Ask to start now
    read -p "Would you like to start the server now? (y/n) " -n 1 -r
    echo ""
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        echo ""
        echo "Starting server..."
        ./run.sh
    fi

else
    echo ""
    print_header "  Installation failed!"
    echo ""
    echo "Trying alternative installation method..."
    echo ""

    # Try with legacy peer deps
    npm install --legacy-peer-deps --yes --no-audit --no-fund

    if [ $? -eq 0 ]; then
        print_success "Installation successful with legacy mode!"
    else
        print_error "All installation methods failed."
        echo ""
        echo "Manual steps:"
        echo "1. Open terminal in this directory"
        echo "2. Run: npm install"
        echo ""
        exit 1
    fi
fi

echo ""
echo "Setup complete! Press Enter to exit..."
read
