# WhatsApp Server Troubleshooting Guide

## Problem: Server Stuck at "Initializing WhatsApp Web.js client..."

### Common Causes & Solutions

#### 1. **Chrome/Chromium Not Found**
The server needs Chrome/Chromium to run WhatsApp Web.

**Solution:**
- **Windows**: Install Chrome from https://www.google.com/chrome/
- **Linux**: Run `sudo apt-get install -y chromium-browser`
- **Mac**: Install Chrome from https://www.google.com/chrome/

#### 2. **Puppeteer Not Downloaded**
Sometimes puppeteer's Chrome download fails.

**Solution:**
```bash
# In the whatsapp-server folder, run:
npm install puppeteer --force
```

#### 3. **Previous Session Corrupted**
Old session data can cause issues.

**Solution:**
```bash
# Delete the auth folder and restart:
# Windows:
rmdir /s .wwebjs_auth

# Linux/Mac:
rm -rf .wwebjs_auth
```

#### 4. **Network/Firewall Issues**
Firewall may block WhatsApp Web connection.

**Solution:**
- Temporarily disable firewall and try again
- Make sure ports 3000 and 443 are not blocked
- Check antivirus settings

---

## Problem: QR Code Not Showing

### Try These Steps:

**Step 1: Check if QR code is being generated**
Look for this message in terminal:
```
QR CODE RECEIVED - SCAN THIS WITH YOUR WHATSAPP APP
```

**Step 2: Use browser to view QR**
1. Keep server running
2. Open browser and go to: `http://localhost:3000/qr`
3. You should see QR code there

**Step 3: Terminal may not support QR display**
Some terminals don't render ASCII QR codes properly.

**Solution:**
- Use the browser method above
- Or use Windows Terminal (recommended for Windows)
- Or use iTerm2 (recommended for Mac)

---

## Problem: Web App Shows "Cannot connect to server"

### Checklist:

1. **Is server running?**
   - Check if you see "Server running on: http://localhost:3000" in terminal
   - Test: Open browser to `http://localhost:3000/health`
   - You should see: `{"status":"ok","message":"WhatsApp server is running"}`

2. **Is server on the same computer as web browser?**
   - The web app must be accessed from the SAME computer where server runs
   - `localhost:3000` only works locally

3. **Is port 3000 already in use?**
   ```bash
   # Windows:
   netstat -ano | findstr :3000

   # Linux/Mac:
   lsof -i :3000
   ```

   If port is busy, edit `server.js` and change `PORT = 3000` to another port like `3001`.

4. **Is there a CORS issue?**
   - Clear browser cache (Ctrl+Shift+Delete)
   - Try opening web app in incognito/private window
   - Make sure you're using `http://` not `https://` for localhost

---

## Step-by-Step Testing Process

### Test 1: Check Server Starts
```bash
cd whatsapp-server
node server.js
```

**Expected output:**
```
============================================================
WhatsApp Web.js Self-Hosted Server
============================================================
Server running on: http://localhost:3000
Health check: http://localhost:3000/health
============================================================
```

**If this doesn't appear:** There's a Node.js or dependency issue.
- Run: `npm install`
- Make sure Node.js version is 16 or higher: `node --version`

---

### Test 2: Check Health Endpoint
Open browser to: `http://localhost:3000/health`

**Expected:**
```json
{"status":"ok","message":"WhatsApp server is running","timestamp":"..."}
```

**If you get error:** Server isn't running or port 3000 is blocked.

---

### Test 3: Check WhatsApp Initialization
Wait 30-60 seconds after starting server.

**Expected:** QR code appears in terminal OR message says to check `/qr` endpoint

**If stuck:** Try solutions in section "Server Stuck at Initializing" above.

---

### Test 4: View QR in Browser
While server is running, open: `http://localhost:3000/qr`

**Expected:** Beautiful QR code page with instructions

**If you see "Generating QR Code..." page:**
- Wait 1-2 minutes
- Refresh page
- Check terminal for errors

---

### Test 5: Scan QR Code
1. Open WhatsApp on phone
2. Go to: Settings → Linked Devices → Link a Device
3. Scan the QR code from terminal or browser
4. Wait for "SUCCESS!" message

**Expected in terminal:**
```
============================================================
SUCCESS! WhatsApp Client is ready!
============================================================
```

---

### Test 6: Connect from Web App
1. Make sure server is running and WhatsApp is connected (from Test 5)
2. Open web app: `https://tinytoolpro.com/whatsapp-bulk-sender/app.php`
3. Click "Self-Hosted" tab
4. Click "Connect to Local Server" button

**Expected:** Green success message "Connected successfully!"

---

## Advanced Troubleshooting

### Enable Debug Logging

Edit `server.js` and add this at the top:
```javascript
process.env.DEBUG = 'puppeteer:*';
```

This will show detailed Puppeteer logs.

---

### Try Simplified Server

If `server.js` doesn't work, try `working-small-server.js`:

```bash
node working-small-server.js
```

This opens Chrome in visible mode which helps debug issues.

---

### Check Dependencies

Make sure all packages are installed:
```bash
npm list --depth=0
```

Should show:
- whatsapp-web.js
- express
- cors
- qrcode
- qrcode-terminal
- multer

---

### Clean Install

If nothing works, do a clean reinstall:

```bash
# Delete everything
rm -rf node_modules package-lock.json .wwebjs_auth

# Reinstall
npm install

# Try again
node server.js
```

---

## Still Having Issues?

### Common Error Messages:

**"Error: Could not find browser"**
→ Install Chrome/Chromium (see solution #1 above)

**"Error: Failed to launch chrome"**
→ Try running: `npm install puppeteer --force`

**"EADDRINUSE: address already in use"**
→ Port 3000 is busy. Change PORT in server.js or kill the process using port 3000.

**"TimeoutError: Waiting for selector..."**
→ Network issue or WhatsApp Web changed. Update whatsapp-web.js: `npm update whatsapp-web.js`

---

## Getting More Help

1. Check the full error message in terminal
2. Check if there are any warnings before the error
3. Try the clean install steps above
4. Make sure your system meets requirements:
   - Node.js 16 or higher
   - 2GB free RAM
   - Internet connection
   - Chrome/Chromium installed

---

## Quick Command Reference

```bash
# Start server
node server.js

# Start with visible Chrome (for debugging)
node working-small-server.js

# Check Node version
node --version

# Reinstall dependencies
npm install

# Delete old session
rm -rf .wwebjs_auth

# Check if port 3000 is free (Linux/Mac)
lsof -i :3000

# Check if port 3000 is free (Windows)
netstat -ano | findstr :3000

# View server logs in real-time
node server.js 2>&1 | tee server.log
```
