# Image Attachment Fix - Self-Hosted Server

## Problem Fixed

**Before:** When sending photos/images through the self-hosted server, they were being sent as "files" (documents) instead of appearing as images in WhatsApp.

**After:** Images now send as proper images (display inline in chat), while other files (PDFs, documents, etc.) still send as file attachments.

## What Was Changed

### File: `server.js`

**Location:** Lines 772-788 (in the `/api/send-file` endpoint)

**Changes Made:**

1. **Added MIME type detection:**
   ```javascript
   // Detect if it's an image based on MIME type
   const isImage = file.mimetype && file.mimetype.startsWith('image/');
   ```

2. **Conditional sending based on file type:**
   ```javascript
   // For images, send without sendMediaAsDocument option (sends as image)
   // For other files, explicitly send as document
   const options = {
       caption: caption || ''
   };

   // If it's NOT an image, send as document explicitly
   if (!isImage) {
       options.sendMediaAsDocument = true;
   }
   ```

## How It Works

### Image Files (jpg, png, gif, webp, etc.)
- Detected by MIME type starting with `image/`
- Sent **without** `sendMediaAsDocument` flag
- Result: Displays as image in WhatsApp chat (preview visible)

### Other Files (pdf, doc, zip, etc.)
- Not detected as images
- Sent **with** `sendMediaAsDocument: true` flag
- Result: Displays as file attachment with download icon

## Supported Image Formats

The fix automatically detects and handles these image types:

- ✅ JPEG/JPG (`image/jpeg`)
- ✅ PNG (`image/png`)
- ✅ GIF (`image/gif`)
- ✅ WebP (`image/webp`)
- ✅ BMP (`image/bmp`)
- ✅ SVG (`image/svg+xml`)
- ✅ Any other format with MIME type starting with `image/`

## No Changes Needed in App

The fix is entirely on the server side. Your main app (`app.php` and JavaScript) doesn't need any changes. The server automatically:

1. Receives the file from your app
2. Checks if it's an image
3. Sends it appropriately to WhatsApp

## Testing the Fix

### To Test:

1. **Stop the server** if it's running (Ctrl+C in terminal)

2. **Restart the server:**
   ```bash
   node server.js
   ```
   Or use your start script:
   ```bash
   ./run.sh          # Linux/Mac
   run.bat           # Windows
   ```

3. **Send a test image:**
   - Connect to WhatsApp (scan QR if needed)
   - Go to your app (app.php)
   - Select Single/Multiple or Bulk mode
   - Attach an image file (jpg, png, etc.)
   - Enter a phone number
   - Send message

4. **Check WhatsApp on phone:**
   - Image should appear inline (not as file)
   - Caption should appear below image
   - Image should be clickable to view full size

### Expected Results:

| File Type | How It Sends | What You See |
|-----------|--------------|--------------|
| Photo.jpg | As Image | 🖼️ Image preview + caption |
| Document.pdf | As Document | 📄 File icon + filename |
| Video.mp4 | As Document* | 🎬 File icon + filename |

*Note: Video support can be added if needed

## Troubleshooting

### Images still sending as files?

1. **Make sure you restarted the server** after applying the fix
2. **Check terminal logs** - should show:
   ```
   [SEND FILE] File type: image/jpeg, Is image: true
   ```
3. **Verify file is actually an image** - try sending a different image

### No logs appearing?

- The server might not have restarted properly
- Run `Ctrl+C` to stop, then start again

## Technical Details

### Why This Happened

The original code used:
```javascript
const result = await client.sendMessage(chatId, media, {
    caption: caption || ''
});
```

This sends everything as a document by default in whatsapp-web.js.

### The Fix

By checking the MIME type and **not** setting `sendMediaAsDocument` for images, WhatsApp Web treats them as images:

```javascript
const options = { caption: caption || '' };

// Only set sendMediaAsDocument for non-images
if (!isImage) {
    options.sendMediaAsDocument = true;
}

const result = await client.sendMessage(chatId, media, options);
```

### WhatsApp Web.js Behavior

- **Without `sendMediaAsDocument`:** WhatsApp determines type based on file (images → as images)
- **With `sendMediaAsDocument: true`:** Everything sends as file/document
- **With `sendMediaAsDocument: false`:** Forces sending as media (but may fail for non-media)

Our fix uses the smart approach: let images be auto-detected, force documents as documents.

## Additional Notes

- This fix is compatible with all providers (Green-API, Whapi, etc.) when they use self-hosted
- No breaking changes to existing functionality
- File size limits still apply (default 50MB)
- The fix includes detailed logging for debugging

## Need More Help?

See also:
- `QUICK-START.md` - How to run the server
- `BROWSER-TAB-INFO.txt` - About keeping browser windows open
- `TROUBLESHOOTING.md` - Common issues and solutions

---

**Fix Applied:** 2025-10-24
**Affects:** Self-Hosted WhatsApp Server (whatsapp-server/server.js)
**Status:** ✅ Ready to Use
