const { Client, LocalAuth, MessageMedia } = require('whatsapp-web.js');
const qrcode = require('qrcode-terminal');
const QRCode = require('qrcode');
const express = require('express');
const cors = require('cors');
const multer = require('multer');
const fs = require('fs');
const path = require('path');
const { exec } = require('child_process');

const app = express();
const PORT = 3000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// File upload configuration
const upload = multer({
    dest: 'uploads/',
    limits: { fileSize: 50 * 1024 * 1024 } // 50MB limit
});

// WhatsApp Client
let client = null;
let isClientReady = false;
let currentQR = null;
let currentQRImage = null;
let clientStatus = 'disconnected';
let qrBrowserOpened = false;

// Initialize WhatsApp Client
function initializeClient() {
    console.clear();
    console.log('🔥 STARTING WHATSAPP SERVER...\n');

    // Simple Chrome detection
    let chromePath = 'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe';
    if (!fs.existsSync(chromePath)) {
        chromePath = 'C:\\Program Files (x86)\\Google\\Chrome\\Application\\chrome.exe';
    }
    if (!fs.existsSync(chromePath)) {
        chromePath = null;
        console.log('⚠️  Chrome not found, using Puppeteer Chromium\n');
    } else {
        console.log('✅ Chrome found:', chromePath, '\n');
    }

    // SIMPLE CONFIG - NO AUTH STRATEGY (back to working version)
    const clientConfig = {
        puppeteer: {
            headless: false,
            args: [
                '--no-sandbox',
                '--disable-setuid-sandbox',
                '--disable-dev-shm-usage'
            ]
        }
    };

    if (chromePath) {
        clientConfig.puppeteer.executablePath = chromePath;
    }

    console.log('📱 Creating WhatsApp client...\n');
    client = new Client(clientConfig);

    // QR Code Generation
    client.on('qr', (qr) => {
        console.log('\n🎉 QR CODE RECEIVED!\n');

        currentQR = qr;
        clientStatus = 'qr_ready';

        // Show in terminal
        qrcode.generate(qr, { small: false });

        // Generate image
        QRCode.toDataURL(qr, { width: 400 }).then(img => {
            currentQRImage = img;
            console.log('\n✅ QR IMAGE READY! Length:', img.length);
            console.log('🌐 Opening browser...\n');

            if (!qrBrowserOpened) {
                qrBrowserOpened = true;
                const url = `http://localhost:${PORT}/qr`;
                const cmd = process.platform === 'win32' ? `start "" "${url}"` : `open "${url}"`;
                exec(cmd);
            }
        });
    });

    client.on('ready', () => {
        console.log('\n✅ CONNECTED!\n');
        isClientReady = true;
        clientStatus = 'connected';
        currentQR = null;
        currentQRImage = null;
        qrBrowserOpened = false;
    });

    client.on('authenticated', () => {
        console.log('✅ Authenticated!\n');
        clientStatus = 'authenticated';
    });

    client.on('auth_failure', (msg) => {
        console.error('❌ Auth failed:', msg);
        clientStatus = 'auth_failed';
    });

    client.on('disconnected', () => {
        console.log('❌ Disconnected\n');
        isClientReady = false;
        clientStatus = 'disconnected';
    });

    console.log('⏳ Initializing client...\n');
    client.initialize();
}

// API Routes

// Health Check
app.get('/health', (req, res) => {
    res.json({
        status: 'ok',
        message: 'WhatsApp server is running',
        timestamp: new Date().toISOString()
    });
});

// Get Status
app.get('/api/status', (req, res) => {
    res.json({
        status: clientStatus,
        ready: isClientReady,
        hasQR: currentQR !== null,
        hasQRImage: currentQRImage !== null,
        qrLength: currentQR ? currentQR.length : 0
    });
});

// Debug endpoint
app.get('/api/debug', (req, res) => {
    res.json({
        server: {
            port: PORT,
            uptime: process.uptime(),
            nodeVersion: process.version,
            platform: process.platform
        },
        whatsapp: {
            clientExists: client !== null,
            clientStatus: clientStatus,
            isReady: isClientReady,
            hasQRData: currentQR !== null,
            hasQRImage: currentQRImage !== null,
            qrDataLength: currentQR ? currentQR.length : 0,
            browserOpened: qrBrowserOpened
        },
        tips: [
            'Check if Chrome window opened to web.whatsapp.com',
            'Look for QR code in the Chrome window (not localhost)',
            'If stuck, delete .wwebjs_auth folder and restart',
            'QR event must fire for localhost page to work'
        ]
    });
});

// Get QR Code (JSON)
app.get('/api/qr', (req, res) => {
    console.log(`[API QR REQUEST] Status: ${clientStatus}, Has QR Image: ${!!currentQRImage}, Image Length: ${currentQRImage ? currentQRImage.length : 0}`);

    if (currentQRImage) {
        res.json({
            success: true,
            qr: currentQR,
            qrImage: currentQRImage,
            status: clientStatus
        });
    } else if (currentQR) {
        // QR exists but image not generated - rare case
        res.json({
            success: false,
            message: 'QR generating, please wait...',
            status: clientStatus
        });
    } else {
        res.json({
            success: false,
            message: clientStatus === 'connected' ? 'Already connected' : 'QR code not available yet',
            status: clientStatus
        });
    }
});

// Direct QR Image endpoint (returns just the image)
app.get('/api/qr-image', (req, res) => {
    console.log(`[QR IMAGE REQUEST] Has Image: ${!!currentQRImage}`);

    if (currentQRImage) {
        // Extract base64 data and send as image
        const base64Data = currentQRImage.replace(/^data:image\/png;base64,/, '');
        const imgBuffer = Buffer.from(base64Data, 'base64');

        res.writeHead(200, {
            'Content-Type': 'image/png',
            'Content-Length': imgBuffer.length
        });
        res.end(imgBuffer);
    } else {
        res.status(404).json({
            success: false,
            message: 'QR code not ready yet'
        });
    }
});

// View QR Code in Browser (HTML)
app.get('/qr', (req, res) => {
    console.log(`[QR PAGE REQUEST] Status: ${clientStatus}, Has QR: ${!!currentQRImage}`);

    if (currentQRImage) {
        console.log('[QR PAGE] Serving QR code image');
        res.send(`
<!DOCTYPE html>
<html>
<head>
    <title>WhatsApp QR Code</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
            color: #fff;
            padding: 20px;
        }
        .container {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            text-align: center;
            max-width: 500px;
            animation: slideIn 0.5s ease-out;
        }
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        h1 {
            color: #333;
            margin: 0 0 10px 0;
            font-size: 32px;
        }
        .subtitle {
            color: #666;
            margin: 0 0 20px 0;
            font-size: 16px;
        }
        .qr-container {
            background: white;
            padding: 20px;
            border-radius: 15px;
            display: inline-block;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            margin: 20px 0;
        }
        img {
            display: block;
            max-width: 100%;
            height: auto;
        }
        .instructions {
            color: #555;
            line-height: 1.8;
            margin: 20px 0;
            text-align: left;
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
        }
        .instructions ol {
            margin: 10px 0;
            padding-left: 25px;
        }
        .instructions li {
            margin: 8px 0;
        }
        .status {
            background: #25D366;
            color: white;
            padding: 10px 20px;
            border-radius: 25px;
            display: inline-block;
            font-weight: bold;
            margin: 10px 0;
            animation: pulse 2s ease-in-out infinite;
        }
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }
        .refresh-btn {
            background: #25D366;
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-size: 16px;
            cursor: pointer;
            margin-top: 20px;
            transition: all 0.3s;
        }
        .refresh-btn:hover {
            background: #128C7E;
            transform: scale(1.05);
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📱 WhatsApp QR Code</h1>
        <p class="subtitle">Scan to connect your WhatsApp</p>
        <div class="status">✅ QR Code Ready</div>
        <div class="qr-container">
            <img src="${currentQRImage}" alt="QR Code" />
        </div>
        <div class="instructions">
            <strong>📖 How to scan:</strong>
            <ol>
                <li>Open <strong>WhatsApp</strong> on your phone</li>
                <li>Go to <strong>Settings</strong> → <strong>Linked Devices</strong></li>
                <li>Tap <strong>Link a Device</strong></li>
                <li>Point your phone at the QR code above</li>
            </ol>
        </div>
        <button class="refresh-btn" onclick="location.reload()">🔄 Refresh QR Code</button>
    </div>
    <script>
        console.log('QR Code page loaded successfully');

        // Check connection status every 2 seconds
        function checkStatus() {
            fetch('/api/status')
                .then(res => res.json())
                .then(data => {
                    console.log('Status:', data);
                    if (data.status === 'connected' || data.ready) {
                        // Show success and close window
                        document.body.innerHTML = '<div style="display: flex; align-items: center; justify-content: center; min-height: 100vh; background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);"><div style="background: white; border-radius: 20px; padding: 60px 40px; text-align: center; box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);"><h1 style="color: #25D366; font-size: 48px; margin: 0 0 20px 0;">✅ Connected!</h1><p style="color: #666; font-size: 18px;">WhatsApp is connected. This window will close in 3 seconds...</p></div></div>';
                        setTimeout(() => window.close(), 3000);
                    } else if (!data.hasQR && data.status !== 'qr_ready') {
                        // QR expired, reload
                        console.log('QR expired, reloading...');
                        location.reload();
                    }
                })
                .catch(err => console.error('Error checking status:', err));
        }
        setInterval(checkStatus, 2000);

        // Auto-refresh every 45 seconds if still showing QR
        setTimeout(() => {
            console.log('Auto-refreshing after 45 seconds');
            location.reload();
        }, 45000);
    </script>
</body>
</html>
        `);
    } else if (clientStatus === 'connected') {
        console.log('[QR PAGE] Already connected');
        res.send(`
<!DOCTYPE html>
<html>
<head>
    <title>WhatsApp Connected</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
        }
        .container {
            background: white;
            border-radius: 20px;
            padding: 60px 40px;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }
        h1 { color: #25D366; font-size: 48px; margin: 0 0 20px 0; }
        p { color: #666; font-size: 18px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>✅ Connected!</h1>
        <p>Your WhatsApp is connected and ready to send messages.</p>
        <p><small>This window will close in 3 seconds...</small></p>
    </div>
    <script>
        setTimeout(() => window.close(), 3000);
    </script>
</body>
</html>
        `);
    } else {
        console.log('[QR PAGE] Waiting for QR code to be generated');
        res.send(`
<!DOCTYPE html>
<html>
<head>
    <title>Loading QR Code</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
        }
        .container {
            background: white;
            border-radius: 20px;
            padding: 60px 40px;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 500px;
        }
        .spinner {
            border: 5px solid #f3f3f3;
            border-top: 5px solid #25D366;
            border-radius: 50%;
            width: 60px;
            height: 60px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        h1 { color: #333; margin: 20px 0 10px 0; font-size: 28px; }
        p { color: #666; line-height: 1.6; margin: 10px 0; }
        .status-text { color: #25D366; font-weight: bold; margin: 15px 0; font-size: 18px; }
        .info-box {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
            margin-top: 20px;
            font-size: 14px;
            color: #555;
            text-align: left;
        }
        #qrContainer { display: none; }
        .qr-box {
            background: white;
            padding: 20px;
            border-radius: 15px;
            display: inline-block;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <div id="loadingState">
            <div class="spinner"></div>
            <h1>⏳ Loading QR Code...</h1>
            <p class="status-text" id="statusText">Initializing...</p>
            <div class="info-box">
                <strong>💡 Please wait:</strong><br>
                • Chrome window is opening<br>
                • WhatsApp Web is loading<br>
                • QR code is being generated<br><br>
                <small>This takes 10-30 seconds</small>
            </div>
        </div>
        <div id="qrContainer">
            <h1>📱 Scan This QR Code</h1>
            <div class="qr-box">
                <img id="qrImage" alt="QR Code" style="max-width: 400px; width: 100%;" />
            </div>
            <div class="info-box">
                <strong>📖 How to scan:</strong><br>
                1. Open WhatsApp on your phone<br>
                2. Go to Settings → Linked Devices<br>
                3. Tap "Link a Device"<br>
                4. Scan this QR code
            </div>
        </div>
    </div>
    <script>
        console.log('QR Page loaded - Aggressive polling mode');

        let checkCount = 0;
        const maxChecks = 120; // 2 minutes of checking

        function checkForQR() {
            checkCount++;
            console.log(\`[Check \${checkCount}] Fetching QR status...\`);

            fetch('/api/qr')
                .then(res => res.json())
                .then(data => {
                    console.log('Response:', data);

                    const statusText = document.getElementById('statusText');
                    statusText.textContent = 'Status: ' + (data.status || 'waiting');

                    if (data.success && data.qrImage) {
                        console.log('✅ QR CODE RECEIVED!');

                        // Hide loading, show QR
                        document.getElementById('loadingState').style.display = 'none';
                        document.getElementById('qrContainer').style.display = 'block';
                        document.getElementById('qrImage').src = data.qrImage;

                        // Stop checking
                        clearInterval(pollInterval);

                        // Check for connection every 3 seconds
                        setInterval(() => {
                            fetch('/api/status')
                                .then(r => r.json())
                                .then(s => {
                                    if (s.ready || s.status === 'connected') {
                                        document.body.innerHTML = '<div style="display: flex; align-items: center; justify-content: center; min-height: 100vh; background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);"><div style="background: white; border-radius: 20px; padding: 60px; text-align: center;"><h1 style="color: #25D366; font-size: 48px;">✅ Connected!</h1><p>Closing in 3 seconds...</p></div></div>';
                                        setTimeout(() => window.close(), 3000);
                                    }
                                });
                        }, 3000);

                    } else if (data.status === 'connected') {
                        console.log('Already connected!');
                        location.href = '/qr';
                    } else if (checkCount >= maxChecks) {
                        statusText.textContent = '⚠️ Timeout - Check console';
                        statusText.style.color = 'red';
                    }
                })
                .catch(err => {
                    console.error('Error:', err);
                });
        }

        // Check immediately
        checkForQR();

        // Then check every 1 second
        const pollInterval = setInterval(checkForQR, 1000);
    </script>
</body>
</html>
        `);
    }
});

// Initialize/Reconnect
app.post('/api/connect', (req, res) => {
    if (isClientReady) {
        res.json({
            success: true,
            message: 'Already connected'
        });
    } else if (client === null) {
        initializeClient();
        res.json({
            success: true,
            message: 'Initializing connection. Please check for QR code.'
        });
    } else {
        res.json({
            success: true,
            message: 'Connection in progress. Please check for QR code.'
        });
    }
});

// Logout/Disconnect
app.post('/api/logout', async (req, res) => {
    try {
        if (client) {
            await client.logout();
            await client.destroy();
            client = null;
            isClientReady = false;
            clientStatus = 'disconnected';
            currentQR = null;
            currentQRImage = null;

            res.json({
                success: true,
                message: 'Logged out successfully'
            });
        } else {
            res.json({
                success: false,
                message: 'No active connection'
            });
        }
    } catch (error) {
        console.error('Logout error:', error);
        res.status(500).json({
            success: false,
            message: 'Error during logout: ' + error.message
        });
    }
});

// Validate client state before operations
async function validateClientState() {
    if (!client || !isClientReady) {
        return { valid: false, message: 'Client not ready' };
    }

    try {
        const state = await client.getState();
        if (state !== 'CONNECTED') {
            isClientReady = false;
            clientStatus = 'disconnected';
            return { valid: false, message: 'Client not connected. Current state: ' + state };
        }
        return { valid: true };
    } catch (error) {
        console.error('State validation error:', error);
        isClientReady = false;
        clientStatus = 'disconnected';
        return { valid: false, message: 'Failed to validate client state' };
    }
}

// Check if number exists on WhatsApp
app.post('/check-number', async (req, res) => {
    try {
        const validation = await validateClientState();
        if (!validation.valid) {
            return res.status(400).json({
                success: false,
                message: validation.message
            });
        }

        const { number } = req.body;

        if (!number) {
            return res.status(400).json({
                success: false,
                message: 'Phone number is required'
            });
        }

        const formattedPhone = number.replace(/\D/g, '');
        const numberId = `${formattedPhone}@c.us`;

        console.log(`[CHECK NUMBER] Checking ${numberId}`);

        const isRegistered = await client.isRegisteredUser(numberId);

        console.log(`[CHECK NUMBER] Result: ${isRegistered}`);

        res.json({
            success: true,
            exists: isRegistered,
            number: formattedPhone
        });

    } catch (error) {
        console.error('Check number error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to check number: ' + error.message,
            exists: false
        });
    }
});

// Send Text Message
app.post('/api/send-message', async (req, res) => {
    try {
        // Validate client state
        const validation = await validateClientState();
        if (!validation.valid) {
            return res.status(400).json({
                success: false,
                message: validation.message
            });
        }

        const { phone, message } = req.body;

        if (!phone || !message) {
            return res.status(400).json({
                success: false,
                message: 'Phone number and message are required'
            });
        }

        // Format phone number (remove non-numeric characters)
        const formattedPhone = phone.replace(/\D/g, '');
        const chatId = `${formattedPhone}@c.us`;

        console.log(`[SEND MSG] Attempting to send to ${chatId}`);

        // Send message with retry logic
        let lastError = null;
        for (let attempt = 1; attempt <= 3; attempt++) {
            try {
                const result = await client.sendMessage(chatId, message);
                console.log(`[SEND MSG] Success on attempt ${attempt}`);

                return res.json({
                    success: true,
                    message: 'Message sent successfully',
                    messageId: result.id.id
                });
            } catch (err) {
                lastError = err;
                console.error(`[SEND MSG] Attempt ${attempt} failed:`, err.message);

                // If session closed, break immediately
                if (err.message.includes('Session closed') || err.message.includes('page has been closed')) {
                    console.error('[SEND MSG] Session closed - marking client as not ready');
                    isClientReady = false;
                    clientStatus = 'disconnected';
                    break;
                }

                // Wait before retry
                if (attempt < 3) {
                    await new Promise(resolve => setTimeout(resolve, 1000));
                }
            }
        }

        throw lastError;

    } catch (error) {
        console.error('Send message error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to send message: ' + error.message
        });
    }
});

// Send File with Caption
app.post('/api/send-file', upload.single('file'), async (req, res) => {
    try {
        // Validate client state
        const validation = await validateClientState();
        if (!validation.valid) {
            // Clean up uploaded file
            if (req.file && req.file.path) {
                try {
                    fs.unlinkSync(req.file.path);
                } catch (e) {
                    console.error('Error deleting temp file:', e);
                }
            }

            return res.status(400).json({
                success: false,
                message: validation.message
            });
        }

        const { phone, caption } = req.body;
        const file = req.file;

        if (!phone || !file) {
            // Clean up uploaded file
            if (file && file.path) {
                try {
                    fs.unlinkSync(file.path);
                } catch (e) {
                    console.error('Error deleting temp file:', e);
                }
            }

            return res.status(400).json({
                success: false,
                message: 'Phone number and file are required'
            });
        }

        // Format phone number
        const formattedPhone = phone.replace(/\D/g, '');
        const chatId = `${formattedPhone}@c.us`;

        console.log(`[SEND FILE] Attempting to send to ${chatId}`);

        // Detect if it's an image based on MIME type
        const isImage = file.mimetype && (
            file.mimetype.startsWith('image/') &&
            !file.mimetype.includes('svg')  // SVG should be sent as document
        );

        console.log(`[SEND FILE] File type: ${file.mimetype}, Is image: ${isImage}, Original name: ${file.originalname}`);

        // Create media from file with proper MIME type
        const media = MessageMedia.fromFilePath(file.path);

        // Ensure MIME type is set correctly
        if (!media.mimetype && file.mimetype) {
            media.mimetype = file.mimetype;
        }

        // Set filename for documents
        if (!isImage && file.originalname) {
            media.filename = file.originalname;
        }

        // Send message with media and retry logic
        let lastError = null;
        for (let attempt = 1; attempt <= 3; attempt++) {
            try {
                const options = {};

                // Add caption if provided
                if (caption) {
                    options.caption = caption;
                }

                // For images: DO NOT set sendMediaAsDocument (default is false, sends as image)
                // For other files: explicitly send as document
                if (!isImage) {
                    options.sendMediaAsDocument = true;
                }

                console.log(`[SEND FILE] Sending with options:`, options);

                const result = await client.sendMessage(chatId, media, options);

                console.log(`[SEND FILE] Success on attempt ${attempt}`);

                // Delete temporary file
                fs.unlinkSync(file.path);

                return res.json({
                    success: true,
                    message: 'File sent successfully',
                    messageId: result.id.id
                });
            } catch (err) {
                lastError = err;
                console.error(`[SEND FILE] Attempt ${attempt} failed:`, err.message);

                // If session closed, break immediately
                if (err.message.includes('Session closed') || err.message.includes('page has been closed')) {
                    console.error('[SEND FILE] Session closed - marking client as not ready');
                    isClientReady = false;
                    clientStatus = 'disconnected';
                    break;
                }

                // Wait before retry
                if (attempt < 3) {
                    await new Promise(resolve => setTimeout(resolve, 1000));
                }
            }
        }

        // Clean up file before throwing error
        try {
            fs.unlinkSync(file.path);
        } catch (e) {
            console.error('Error deleting temp file:', e);
        }

        throw lastError;

    } catch (error) {
        console.error('Send file error:', error);

        // Clean up file if exists
        if (req.file && req.file.path) {
            try {
                fs.unlinkSync(req.file.path);
            } catch (e) {
                console.error('Error deleting temp file:', e);
            }
        }

        res.status(500).json({
            success: false,
            message: 'Failed to send file: ' + error.message
        });
    }
});

// Get Server Info
app.get('/api/info', (req, res) => {
    const info = {
        version: '1.0.0',
        status: clientStatus,
        ready: isClientReady,
        platform: process.platform,
        nodeVersion: process.version,
        uptime: process.uptime()
    };

    if (isClientReady && client) {
        client.getState().then(state => {
            info.whatsappState = state;
            res.json(info);
        }).catch(() => {
            res.json(info);
        });
    } else {
        res.json(info);
    }
});

// Start Server
app.listen(PORT, () => {
    console.log('🚀 Server running: http://localhost:' + PORT);
    console.log('📊 Status: http://localhost:' + PORT + '/api/status');
    console.log('🔍 QR Page: http://localhost:' + PORT + '/qr\n');

    initializeClient();
});

// Graceful Shutdown
process.on('SIGINT', async () => {
    console.log('\nShutting down gracefully...');
    if (client) {
        await client.destroy();
    }
    process.exit(0);
});

process.on('SIGTERM', async () => {
    console.log('\nShutting down gracefully...');
    if (client) {
        await client.destroy();
    }
    process.exit(0);
});
