#!/usr/bin/env node

/**
 * WhatsApp Server Diagnostic Tool
 * Run this to check if your system is ready to run the WhatsApp server
 */

const fs = require('fs');
const path = require('path');
const http = require('http');

console.log('='.repeat(70));
console.log('WhatsApp Server Diagnostic Tool');
console.log('='.repeat(70));
console.log('');

let allGood = true;

// Check 1: Node.js version
console.log('✓ Checking Node.js version...');
const nodeVersion = process.version;
const majorVersion = parseInt(nodeVersion.split('.')[0].substring(1));
console.log(`  Node.js version: ${nodeVersion}`);
if (majorVersion < 16) {
    console.log('  ❌ ERROR: Node.js 16 or higher required!');
    console.log('  → Download from: https://nodejs.org/');
    allGood = false;
} else {
    console.log('  ✅ Version is good!');
}
console.log('');

// Check 2: Required files
console.log('✓ Checking required files...');
const requiredFiles = ['server.js', 'package.json'];
let filesOk = true;
for (const file of requiredFiles) {
    if (fs.existsSync(file)) {
        console.log(`  ✅ ${file} - Found`);
    } else {
        console.log(`  ❌ ${file} - Missing!`);
        filesOk = false;
        allGood = false;
    }
}
if (filesOk) {
    console.log('  ✅ All required files present');
}
console.log('');

// Check 3: node_modules
console.log('✓ Checking dependencies...');
if (fs.existsSync('node_modules')) {
    console.log('  ✅ node_modules folder exists');

    // Check key packages
    const packages = ['express', 'whatsapp-web.js', 'qrcode', 'cors'];
    let packagesOk = true;
    for (const pkg of packages) {
        const pkgPath = path.join('node_modules', pkg);
        if (fs.existsSync(pkgPath)) {
            console.log(`  ✅ ${pkg} - Installed`);
        } else {
            console.log(`  ❌ ${pkg} - Missing!`);
            packagesOk = false;
            allGood = false;
        }
    }

    if (!packagesOk) {
        console.log('  → Run: npm install');
    }
} else {
    console.log('  ❌ node_modules not found!');
    console.log('  → Run: npm install');
    allGood = false;
}
console.log('');

// Check 4: Port 3000 availability
console.log('✓ Checking port 3000...');
const server = http.createServer();
server.once('error', (err) => {
    if (err.code === 'EADDRINUSE') {
        console.log('  ❌ Port 3000 is already in use!');
        console.log('  → Kill the process using port 3000 or change PORT in server.js');
        allGood = false;
    } else {
        console.log('  ⚠️  Port check failed:', err.message);
    }
});
server.once('listening', () => {
    console.log('  ✅ Port 3000 is available');
    server.close();
});
server.listen(3000);

// Wait for port check to complete
setTimeout(() => {
    console.log('');

    // Check 5: Chrome browser
    console.log('✓ Checking for Chrome browser...');
    if (process.platform === 'win32') {
        const chromePaths = [
            'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe',
            'C:\\Program Files (x86)\\Google\\Chrome\\Application\\chrome.exe',
            process.env.LOCALAPPDATA + '\\Google\\Chrome\\Application\\chrome.exe'
        ];

        let chromeFound = false;
        for (const chromePath of chromePaths) {
            if (fs.existsSync(chromePath)) {
                console.log(`  ✅ Chrome found at: ${chromePath}`);
                chromeFound = true;
                break;
            }
        }

        if (!chromeFound) {
            console.log('  ⚠️  Chrome not found in standard locations');
            console.log('  → Puppeteer will download Chromium (may take longer on first run)');
        }
    } else {
        console.log('  ℹ️  Not on Windows, skipping Chrome check');
    }
    console.log('');

    // Check 6: Previous session
    console.log('✓ Checking for previous WhatsApp session...');
    if (fs.existsSync('.wwebjs_auth')) {
        console.log('  ℹ️  Previous session found (.wwebjs_auth)');
        console.log('  → If you have connection issues, delete this folder and try again');
    } else {
        console.log('  ℹ️  No previous session (this is normal for first run)');
    }
    console.log('');

    // Check 7: Memory
    console.log('✓ Checking system resources...');
    const totalMem = require('os').totalmem();
    const freeMem = require('os').freemem();
    const usedMem = totalMem - freeMem;
    const totalGB = (totalMem / 1024 / 1024 / 1024).toFixed(2);
    const freeGB = (freeMem / 1024 / 1024 / 1024).toFixed(2);

    console.log(`  Total RAM: ${totalGB} GB`);
    console.log(`  Free RAM: ${freeGB} GB`);

    if (freeMem < 500 * 1024 * 1024) { // Less than 500MB free
        console.log('  ⚠️  Low memory! Close some applications.');
    } else {
        console.log('  ✅ Memory looks good');
    }
    console.log('');

    // Check 8: Internet connection
    console.log('✓ Checking internet connection...');
    const https = require('https');
    https.get('https://web.whatsapp.com', (res) => {
        if (res.statusCode === 200 || res.statusCode === 302) {
            console.log('  ✅ Can reach WhatsApp Web');
        } else {
            console.log(`  ⚠️  Unexpected response from WhatsApp Web (${res.statusCode})`);
        }

        finalReport();
    }).on('error', (err) => {
        console.log('  ❌ Cannot reach WhatsApp Web!');
        console.log(`  Error: ${err.message}`);
        console.log('  → Check your internet connection and firewall');
        allGood = false;

        finalReport();
    });

    function finalReport() {
        console.log('');
        console.log('='.repeat(70));
        if (allGood) {
            console.log('✅ DIAGNOSTIC PASSED!');
            console.log('='.repeat(70));
            console.log('');
            console.log('Your system is ready to run the WhatsApp server.');
            console.log('');
            console.log('Next steps:');
            console.log('  1. Run: node server.js');
            console.log('  2. Wait for QR code (30-60 seconds)');
            console.log('  3. Scan QR code with WhatsApp');
            console.log('  4. Connect from web app');
            console.log('');
        } else {
            console.log('❌ DIAGNOSTIC FAILED!');
            console.log('='.repeat(70));
            console.log('');
            console.log('Please fix the issues above before running the server.');
            console.log('');
            console.log('Common fixes:');
            console.log('  • Missing dependencies? Run: npm install');
            console.log('  • Port in use? Change PORT in server.js');
            console.log('  • Old Node.js? Update from: https://nodejs.org/');
            console.log('  • Session issues? Delete .wwebjs_auth folder');
            console.log('');
        }
        console.log('For more help, see: TROUBLESHOOTING.md');
        console.log('='.repeat(70));
    }
}, 1000);
