#!/bin/bash

# Change to script directory
cd "$(dirname "$0")"

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_header() {
    echo -e "${BLUE}============================================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}============================================================${NC}"
}

print_success() {
    echo -e "${GREEN}[OK]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

clear
echo ""
print_header "  WhatsApp Web.js Self-Hosted Server - Auto Installer"
echo ""
echo "Current Directory: $(pwd)"
echo ""

# Step 1: Check if Node.js is installed
echo "[1/5] Checking Node.js installation..."
if ! command -v node &> /dev/null; then
    print_error "Node.js is NOT installed!"
    echo ""
    echo "Please install Node.js first:"
    echo "  Ubuntu/Debian: sudo apt-get install nodejs npm"
    echo "  CentOS/RHEL:   sudo yum install nodejs npm"
    echo "  macOS:         brew install node"
    echo "  Or download from: https://nodejs.org/"
    echo ""
    exit 1
fi

NODE_VERSION=$(node --version)
print_success "Node.js found: $NODE_VERSION"

# Check Node.js version (minimum v14)
NODE_MAJOR=$(node --version | cut -d'.' -f1 | sed 's/v//')
if [ "$NODE_MAJOR" -lt 14 ]; then
    print_warning "Node.js version is old. Version 14+ recommended."
    print_warning "Current version: $NODE_VERSION"
    echo ""
fi

NPM_VERSION=$(npm --version)
print_success "npm found: v$NPM_VERSION"
echo ""

# Step 2: Check if package.json exists
echo "[2/5] Checking project files..."
if [ ! -f "package.json" ]; then
    print_error "package.json not found!"
    echo "        Please ensure you're in the correct directory."
    echo "        Current directory: $(pwd)"
    echo ""
    exit 1
fi
print_success "package.json found"
echo ""

# Step 3: Clean old installation
echo "[3/5] Cleaning previous installation..."
if [ -d "node_modules" ]; then
    echo "      Removing old node_modules..."
    rm -rf node_modules
    if [ -d "node_modules" ]; then
        print_warning "Could not remove old node_modules completely."
        print_warning "Installation will continue anyway."
    else
        print_success "Old node_modules removed"
    fi
fi

if [ -f "package-lock.json" ]; then
    echo "      Removing old package-lock.json..."
    rm -f package-lock.json
    if [ -f "package-lock.json" ]; then
        print_warning "Could not remove package-lock.json"
    else
        print_success "package-lock.json removed"
    fi
fi
echo ""

# Step 4: Create required directories
echo "[4/5] Creating required directories..."
if [ ! -d "uploads" ]; then
    mkdir -p uploads
    if [ -d "uploads" ]; then
        print_success "Created: uploads/"
    else
        print_warning "Could not create uploads/ directory"
    fi
else
    print_success "uploads/ already exists"
fi

if [ ! -d ".wwebjs_auth" ]; then
    print_success ".wwebjs_auth/ will be created on first run"
else
    print_success ".wwebjs_auth/ already exists"
fi

# Set proper permissions
chmod 755 . 2>/dev/null
chmod 755 uploads 2>/dev/null
echo ""

# Step 5: Install dependencies
echo "[5/5] Installing dependencies..."
echo "      This may take 2-5 minutes depending on your internet speed."
echo "      Please be patient and do NOT interrupt!"
echo ""
print_header ""
echo ""

# Clear npm cache (helps with installation issues)
npm cache clean --force > /dev/null 2>&1

# Install with verbose output
npm install --no-audit --no-fund

# Check installation result
if [ $? -eq 0 ]; then
    echo ""
    print_header ""
    echo ""
    print_success "Installation completed successfully!"
    echo ""
    print_header "  Next Steps:"
    echo ""
    echo "  1. Run ./run.sh to start the server"
    echo "  2. Scan the QR code with WhatsApp mobile app"
    echo "  3. Use the web interface to send messages"
    echo ""
    print_header "  Quick Start Command:"
    echo ""
    echo "  ./run.sh"
    echo ""
    print_header ""
    echo ""

    # Verify critical dependencies
    echo "Verifying installation..."
    if [ -d "node_modules/whatsapp-web.js" ]; then
        print_success "whatsapp-web.js installed"
    else
        print_warning "whatsapp-web.js may not be installed correctly"
    fi

    if [ -d "node_modules/express" ]; then
        print_success "express installed"
    else
        print_warning "express may not be installed correctly"
    fi

    if [ -d "node_modules/qrcode-terminal" ]; then
        print_success "qrcode-terminal installed"
    fi
    echo ""

    # Make scripts executable
    echo "Setting file permissions..."
    chmod +x run.sh 2>/dev/null
    chmod +x install.sh 2>/dev/null
    print_success "Scripts are now executable"
    echo ""

else
    echo ""
    print_header "[ERROR] Installation FAILED!"
    echo ""
    echo "Common solutions:"
    echo "  1. Check your internet connection"
    echo "  2. Run with sudo if permission denied"
    echo "  3. Clear npm cache: npm cache clean --force"
    echo "  4. Update npm: sudo npm install -g npm@latest"
    echo "  5. Install build tools:"
    echo "     Ubuntu/Debian: sudo apt-get install build-essential"
    echo "     CentOS/RHEL:   sudo yum groupinstall 'Development Tools'"
    echo "     macOS:         xcode-select --install"
    echo ""
    echo "If problem persists:"
    echo "  - Delete node_modules folder: rm -rf node_modules"
    echo "  - Run this installer again: ./install.sh"
    echo ""
    print_header ""
    echo ""
    exit 1
fi

# Offer to create an alias (optional)
echo ""
read -p "Would you like to add a 'whatsapp-server' command alias? (y/n) " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    SHELL_RC=""
    if [ -f "$HOME/.bashrc" ]; then
        SHELL_RC="$HOME/.bashrc"
    elif [ -f "$HOME/.zshrc" ]; then
        SHELL_RC="$HOME/.zshrc"
    fi

    if [ -n "$SHELL_RC" ]; then
        ALIAS_CMD="alias whatsapp-server='cd $(pwd) && ./run.sh'"
        if ! grep -q "whatsapp-server" "$SHELL_RC"; then
            echo "" >> "$SHELL_RC"
            echo "# WhatsApp Server alias" >> "$SHELL_RC"
            echo "$ALIAS_CMD" >> "$SHELL_RC"
            print_success "Alias added to $SHELL_RC"
            echo "        Run 'source $SHELL_RC' or restart your terminal"
            echo "        Then you can use: whatsapp-server"
        else
            print_info "Alias already exists in $SHELL_RC"
        fi
    fi
fi

echo ""
echo "Installation complete! Press Enter to exit..."
read
