#!/bin/bash

# Change to script directory
cd "$(dirname "$0")"

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Function to print colored output
print_header() {
    echo -e "${BLUE}============================================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}============================================================${NC}"
}

print_success() {
    echo -e "${GREEN}[OK]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_info() {
    echo -e "${CYAN}[INFO]${NC} $1"
}

clear
echo ""
print_header "  WhatsApp Web.js Self-Hosted Server"
echo ""

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    print_error "Node.js is not installed!"
    echo "        Please run ./install.sh first"
    echo ""
    exit 1
fi

# Check if node_modules exists
if [ ! -d "node_modules" ]; then
    print_error "Dependencies not installed!"
    echo ""
    echo "Please run the installer first:"
    echo "  1. Run: ./install.sh"
    echo "  2. Wait for installation to complete"
    echo "  3. Run this script again"
    echo ""
    read -p "Would you like to run the installer now? (y/n) " -n 1 -r
    echo ""
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        echo ""
        echo "Starting installer..."
        ./install.sh
        if [ $? -eq 0 ]; then
            echo ""
            echo "Installation complete! Starting server..."
            sleep 3
        else
            echo ""
            echo "Installation failed. Please check the errors above."
            exit 1
        fi
    else
        exit 1
    fi
fi

# Check if server.js exists
if [ ! -f "server.js" ]; then
    print_error "server.js not found!"
    echo "        Please ensure all files are in the correct directory."
    echo "        Current directory: $(pwd)"
    echo ""
    exit 1
fi

# Create uploads directory if it doesn't exist
if [ ! -d "uploads" ]; then
    print_info "Creating uploads directory..."
    mkdir -p uploads
fi

# Set permissions
chmod 755 uploads 2>/dev/null

echo "Current Directory: $(pwd)"
echo ""
print_header "  Server Information"
echo ""
echo "  Server URL:     http://localhost:3000"
echo "  Health Check:   http://localhost:3000/health"
echo "  API Status:     http://localhost:3000/api/status"
echo ""
print_header "  Important Instructions"
echo ""
echo "  1. Wait for QR code to appear in this window"
echo "  2. Open WhatsApp on your phone"
echo "  3. Go to Settings > Linked Devices"
echo "  4. Tap \"Link a Device\""
echo "  5. Scan the QR code shown below"
echo ""
echo "  After scanning:"
echo "  - Go to your web interface"
echo "  - Select \"Self-Hosted\" option"
echo "  - Click \"Connect\" button"
echo "  - Start sending messages!"
echo ""
print_header "  Press Ctrl+C to stop the server"
echo ""
echo "Starting server..."
echo ""

# Function to handle cleanup on exit
cleanup() {
    echo ""
    print_header "Server has stopped."
    echo ""
    exit 0
}

# Trap Ctrl+C and cleanup
trap cleanup SIGINT SIGTERM

# Start the server
node server.js

# If server stops normally, show message
cleanup
